module net.BurtonRadons.dedit.highlight.cecil;

import net.BurtonRadons.dedit.main;
import std.path;
import std.string;
import std.ctype;

alias std.ctype.isdigit isdigit;

/** The Cecil highlighter. */
class Cecil_Highlighter : SyntaxHighlighter
{
    char [] [] reservedBase = 
    [
        "above", "abstract", "any", "array", "assert",
        "bool", 
        "concrete",  
        "dynamic", 
        "extend", "extends",
        "field", "for", 
        "if", "implementation", "int", "isa", 
        "let", 
        "method", "module", 
        "object",
        "precedence", "private", "put",
        "representation", 
        "signature", "subtypes", 
        "template", "type",
        "var", "void",
    ];

    char [] [] preprocBase =
    [
        "define",
        "endif",
        "else",
        "error",
        "if",
        "ifdef",
        "ifndef",
        "include",
        "pragma",
        "undef",
    ];

    char [] [] ifPreprocBase =
    [
        "defined",
        "pragma",
    ];

    char [] [] pragmaPreprocBase =
    [
        "once",
    ];

    bit [char []] reserved;
    bit [char []] preproc;
    bit [char []] ifPreproc;
    bit [char []] pragmaPreproc;

    const char [] symbols = "()[]<>{}:;=!%^&*-+|/.,$~\\@";

    bit [char []] keywordDict (char [] [] base)
    {
        bit [char []] dict;

        for (int c; c < base.length; c ++)
            dict [base [c]] = true;

        return dict;
    }

    this ()
    {
        reserved = keywordDict (reservedBase);
        preproc = keywordDict (preprocBase);
        ifPreproc = keywordDict (ifPreprocBase);
        pragmaPreproc = keywordDict (pragmaPreprocBase);
    }

    static this ()
    {
        list ~= new Cecil_Highlighter ();
    }

    override char [] name () { return "Cecil"; }
    override char [] exts () { return "*.cecil"; }

    override float match (char [] filename, char [] [] data)
    {
        char [] ext = std.string.tolower (std.path.getExt (filename));

        if (ext == "cecil")
            return 1;
        return 0;
    }

    final bit isSymbol (char f)
    {
        for (int c; c < symbols.length; c ++)
            if (f == symbols [c])
                return true;

        return false;
    }

    final bit isIdentifierStart (char f) { return isalpha (f) || f == '_'; }
    final bit isIdentifierMiddle (char f) { return isalnum (f) || f == '_'; }

    final char [] getKeyword (char *c, int n)
    {
        int d;

        if (n == 0 || !isIdentifierStart (*c))
            return null;
        for (d = 1; d < n; d ++)
            if (!isIdentifierMiddle (c [d]))
                break;

        return c [0 .. d];
    }
    
    struct LineInfo
    {
        char code; /**< Current syntax highlighting code. */
        char open;
            /**< Current open type:
               * <ul>
               * <li>'*' - multiline comment.
               * <li>'"' - double-quoted std.string.
               * <li>"'" - single-quoted std.string.
               * <li>'i' - identifier.
               * <li>'/' - single-line comment.
               * <li>'#' - number.
               * </ul>
               */
    }

    override int extraSize ()
    {
        return LineInfo.size;
    }

    override void highlight (char [] line, char [] high, void *lastp, void *nextp)
    {
        LineInfo *last = (LineInfo *) lastp;
        LineInfo *next = (LineInfo *) nextp;
        char *c, h, e;
        char code, open;
        bit [char []] keywords = reserved;
        bit isInclude;

        if (last !== null)
        {
            code = last.code;
            open = last.open;
        }

        c = line;
        h = high;
        e = c + line.length;

        /* Skip initial space and handle a preprocessor token if there is one. */
        while (c < e && isspace (*c))
            *h ++ = 0, c ++;

        if (c < e && *c == '#')
        {
            *h ++ = 'r', c ++;
            while (c < e)
            {
                if (isspace (*c))
                    *h ++ = 'r', c ++;
                else
                    break;
            }

            char [] r;

            r = getKeyword (c, (int) (e - c));
            h [0 .. r.length] = (r in preproc) ? 'r' : 'i';
            h += r.length;
            c += r.length;
            isInclude = (bit) (r == "include");
            if (r == "if")
                keywords = ifPreproc;
            if (r == "pragma")
                keywords = pragmaPreproc;

            /* Make the rest a std.string. */
            if (r == "error" || r == "warning")
            {
                while (c < e)
                    *h ++ = '"', c ++;
            }
        }

        while (c < e)
        {
            int n = (int) (e - c);
            char f = *c;
            char [] r;

        restart:
            if (open == '*')
            {
                if (n > 2 && c [0] == '-' && c [1] == '-' && c [2] == ')')
                {
                    *h ++ = code;
                    *h ++ = code;
                    *h ++ = code;
                    code = 0;
                    open = 0;
                    c += 2;
                }
                else
                    goto def;
            }
            else if (open == '"')
            {
                if (c [0] == '\\')
                {
                    *h ++ = code;
                    c ++;
                    if (c < e)
                        goto def;
                    goto dun;
                }
                else if (c [0] == '"')
                {
                    *h ++ = code;
                    code = open = 0;
                    c ++;
                }
                else
                    goto def;
            }
            else if (open == '\"')
            {
                if (c [0] == '\\')
                {
                    *h ++ = code;
                    c ++;
                    if (c < e)
                        goto def;
                    goto dun;
                }
                else if (c [0] == '\'')
                {
                    *h ++ = code;
                    code = open = 0;
                    c ++;
                }
                else
                    goto def;
            }
            else if (open == 'i')
            {
                if (isalnum (f) || f == '_')
                    goto def;
                open = code = 0;
                goto restart;
            }
            else if (open == '/')
                goto def;
            else if (open == '#')
            {
                if (isdigit (f) || f == 'x' || f == 'X' || f == '.' || f == 'e' || f == 'E'
                 || f == 'a' || f == 'b' || f == 'c' || f == 'd' || f == 'e' || f == 'f'
                 || f == 'A' || f == 'B' || f == 'C' || f == 'D' || f == 'E' || f == 'F'
                 || f == 'l' || f == 'L')
                    goto def;
                else
                {
                    code = open = 0;
                    goto restart;
                }
            }
            else if (open == '<')
            {
                *h ++ = code, c ++;
                if (f == '>')
                {
                    code = open = 0;
                    goto dun;
                }
            }
        /* open == 0 from here on */
            else if (isInclude && f == '<')
            {
                open = '<', code = '"';
                goto def;
            }
            else if (n > 2 && c [0] == '(' && c [1] == '-' && c [2] == '-')
            {
                open = code = '*';
                *h ++ = code;
                *h ++ = code;
                c += 2;
            }
            else if (f == '\"')
            {
                open = code = '"';
                goto def;
            }
            else if (f == '\'')
            {
                open = '\''; 
                code = '\"';
                goto def;
            }
            else if (n > 1 && c [0] == '-' && c [1] == '-')
            {
                open = '/';
                code = '*';
                goto def;
            }
            else if (isdigit (f) || (f == '.' && (n == 1 || isdigit (c [1]))))
            {
                open = code = '#';
                goto def;
            }
            else
            {
                if (isSymbol (f))
                {
                    *h ++ = 's';
                    c ++;
                }
                else if ((r = getKeyword (c, n)) !== null)
                {
                    h [0 .. r.length] = (r in keywords) ? 'r' : 'i';
                    c += r.length;
                    h += r.length;
                }
                else
                    goto def;
            }

        dun:
            continue;
        def:
            *h ++ = code;
            c += 1;
        }

        if (open == 'i' || open == '/' || open == '#' || open == '<')
            open = 0;

        next.code = code;
        next.open = open;
    }
}

